/* drone didactique D2C :
ce programme réalise l'asservissement en position angulaire du balancier du drone didactique D2C
à partir de la mesure délivrée par le capteur "angle pivot".
(asservissement à une seule boucle : boucle de position)
le correcteur utilisé pour stabiliser le système est un correcteur à avance de phase 
dont l'équation de récurrence est écrite avec la méthode des rectangles.

Nota : la période d'échantillonnage est fixée à 20 ms (constante "periode_echantillonnage")
par le fait que la bibliothèque "servo" régénère la commande des moteurs toutes les 20 ms.

**** câblage des entrées sorties de l'arduino : ****
broche masse du D2C sur                     GND arduino
broche gyro Ugyr du D2C sur                 entrée analogique A0 arduino  (non utilisée ici)
broche accéléro Uax du D2C sur              entrée analogique A1 arduino  (non utilisée ici)
broche potentio gauche du D2C sur           entrée analogique A2 arduino
broche potentio  droit du D2C sur           entrée analogique A3 arduino
broche "capteur angle pivot" sur            entrée analogique A4 arduino
broche commande du moteur gauche du D2C sur sortie numérique 7 arduino
broche commande du moteur droit du D2C sur  sortie numérique 8 arduino

- tous droits réservés - François WEISS - pour DMS -
*/

//************** variables pour le correcteur *****************************************
long periode_echantillonnage = 20;    // frequence de mise à jour du correcteur (en millisecondes) ; 
float precedente_entree = 0;          // variable globale pour la fonction AVPH (correcteur à avance de phase)
float precedente_sortie = 0;          // variable globale pour la fonction AVPH (correcteur à avance de phase)
float kp1 = 0.004;
float av = 25;
float tau = 0.05;     // reglage du correcteur à avance de phase
           
//****************************************** la fonction pour le correcteur************************************************
float AVPH_out(float entree, long dt) {
  float sortie;
  // cas de la méthode des rectangles pour exprimer le correcteur à avance de phase :   
  sortie = (1/((float(dt))/1000.0f + tau)) * ( tau*precedente_sortie + ((float(dt))/1000.0f + av*tau)*entree - av*tau*precedente_entree );
  //sortie = constrain(sortie,-1023,1023);     // bornage des valeurs
  precedente_entree = entree;                // pour le prochain calcul   
  precedente_sortie = sortie;                // pour le prochain calcul
  return sortie;                             //retourne la sortie du correcteur pour exploitation de la fonction
}

//*************** vitesse du port serie (pour tests et affichage)
unsigned int vitesse_port = 57600;                 // variable pour activer la lecture sur le port série et transmettre des valeurs vers le PC (pour contrôle)
                                                 // à régler à l'identique dans la fenêtre "serial monitor" de l'interface de programmation Arduino (et processing ).
// ******************* variables pour gerer le cycle
unsigned int nb_cycles = 0;                      // variable pour compter le nombre de cycles : utilisé pour ne pas générer des sorties sur le port série à chaque cycle
long temps_debut_cycle = 0;                      // pour la mesure du temps de cycle de l'asservissement
int temps_cycle_calcule = 0;                     // pour la mesure du temps de cycle de l'asservissement
float ecart = 0;                               // différence entre consigne potentiomètre et mesure pour la boucle d'asservissement (de -1023 à + 1023)
float commande = 0;                            // ecart brut traité par le correcteur = commande
float pi=3.141597;

//*********** variables pour la commande manuelle :
int mesure_potentio_gauche = 0;              // variable pour mémoriser la valeur du potentiomètre gauche (commande des gaz)
int mesure_potentio_droit = 512;             // variable pour mémoriser la valeur du potentiomètre droit  (commande de tangage)(initialisée à valeur moyenne : 512)(entrée sur 10 bits = 1024 pts)
int broche_potentio_gauche = 2;              // variable pour déclarer le port 2 comme port où est branché le potentiomètre gauche (commande des gaz)
int broche_potentio_droit = 3;               // variable pour déclarer le port 3 comme port où est branché le potentiomètre droit (commande de tangage)

//*********** variables pour le fonctionnement des moteurs : (voir aussi dans la fonction "setup" pour les broches de sortie)
#include <Servo.h>                    // on utilise la bibliothèque (ou la classe) "Servo"
Servo moteur_brushless_gauche;        // créé un "objet" nommé "moteur_brushless_gauche" dans la classe "Servo"
Servo moteur_brushless_droit;         // créé un "objet" nommé "moteur_brushless_droit" dans la classe "Servo"
float creneau_min = 1100;          // définit la valeur en microsecondes de la largeur d'impulsion minimale pour le contrôleur utilisé ;
float creneau_max = 1700;          // définit la valeur en microsecondes de la largeur d'impulsion maximale pour le contrôleur utilisé ;
                                      // (adaptation à chaque type de moteur+contrôleur en fonction de valeur "creneau_min" et "creneau_max")
float amplitude_max_creneau = creneau_max - creneau_min;
float valeur_nominale_creneau = creneau_min;   //initialisation de variable pour définir la valeur en milliseconde de largeur créneau pour obtenir la vitesse moyenne d'un moteur
float commande_unitarisee = 0.5;               // variable pour ramener la variation de consigne tangage sur une échelle de 0 à 1 (initialement de 0 à 1023 ); initialisée à la valeur milieu
float commande_gaz_unitarisee = 0;             // variable pour ramener la variation de consigne gaz sur une échelle de -1 à 1 (initialement de 0 à 1023 ); initialisée à la valeur minimum
float creneau_gauche = creneau_min;            // variable pour mémoriser la commande à envoyer vers le servo (entre pulsmin et pulsemax)
float creneau_droit = creneau_min;             // variable pour mémoriser la commande à envoyer vers le servo (entre pulsmin et pulsemax)

// ************ variables pour le "capteur angle pivot"
int broche_angle_pivot=4;                        // variable pour déclarer le port 4 comme port où est branché le capteur "angle pivot"
int mesure_angle_pivot_nul = 512 ;               // valeur en bits de la mesure générée par le capteur à l'arrêt
int mesure_angle_pivot = 512; // variable pour récupérer la tension du capteur "angle pivot" (initialisée à 512 environ)

//****************************************************************************************************************************************************
//setup = déclaration des actions nécessaires au démarrage (traité une seule fois au lancement du programme)
void setup(){
  Serial.begin(vitesse_port);                   // mise en route du port série à la vitesse choisie dans "vitesseport"
  moteur_brushless_droit.attach(8);             // utilise la méthode "attach" de la classe Servo pour affecter la commande à la broche numérique 8 de l'arduino
  moteur_brushless_gauche.attach(7);            // utilise la méthode "attach" de la classe Servo pour affecter la commande à la broche numérique 7 de l'arduino
}

//******************************************************************************************************************************************************
//boucle principale
void loop(){
  temps_debut_cycle = micros();                   // pour le calcul et la fixation du temps de cycle (nécessaire au correcteur)

  /* ************* lecture des entrées capteurs **************************************** */
  mesure_angle_pivot=analogRead(broche_angle_pivot);         // acquisition de la tension du capteur "angle pivot" 0 à 5V  (valeurs de 0 à 1023 sur 10 bits)

  // ********************************   asservissement ************************************
  //******************traitement de la commande du tangage
  mesure_potentio_droit = analogRead(broche_potentio_droit);                                         // acquisition de la tension du potentio 0 à 5V sur 10 bits (valeurs de 0 à 1023 )
  ecart = float(mesure_potentio_droit - 512) - float(mesure_angle_pivot - mesure_angle_pivot_nul);   // calcul de l'écart entre consigne et mesure 
  commande = AVPH_out(kp1*ecart, periode_echantillonnage);                                            // application de la fonction du correcteur à avance de phase 
  //commande_unitarisee = commande / 1023.0f; 
  commande_unitarisee = commande / 50.0f;                                                             // rééchelonnement de la commande de tangage 

  //********************* traitement de la commande des gaz :
  mesure_potentio_gauche = analogRead(broche_potentio_gauche);                // acquisition de la tension du potentio gauche à 5V sur 10 bits (valeurs de 0 à 1023 sur 10 bits)
  commande_gaz_unitarisee = float(mesure_potentio_gauche) / 1023.0f;  // rééchelonnement de la commande de gaz : valeur de la commande (0 à 1023) ramenée sur une échelle de 0 à 1
  valeur_nominale_creneau = creneau_min + (amplitude_max_creneau * commande_gaz_unitarisee) ;  // créneau nominal généré par la commande des gaz à partir de la valeur de potentio

  //******************************** commande des moteurs *****************************************************
  creneau_gauche = valeur_nominale_creneau + (amplitude_max_creneau * commande_unitarisee) ;      // calcul largeur d'impulsion moteur gauche autour de la valeur nominale
  if (creneau_gauche < creneau_min) creneau_gauche = creneau_min ;                                // on limite à des valeurs autorisées par la carte de puissance
  if (creneau_gauche > creneau_max) creneau_gauche = creneau_max ;
  creneau_droit = valeur_nominale_creneau - (amplitude_max_creneau * commande_unitarisee) ;      // calcul largeur d'impulsion moteur droit autour de la valeur nominale
  if (creneau_droit < creneau_min) creneau_droit = creneau_min ;                                 // on limite à des valeurs autorisées par la carte de puissance
  if (creneau_droit > creneau_max) creneau_droit = creneau_max ;
  moteur_brushless_gauche.writeMicroseconds(int(creneau_gauche));   // utilise la méthode "writeMicroseconds" de la classe Servo pour générer les créneaux en sortie pour vitesse moteur gauche
  moteur_brushless_droit.writeMicroseconds(int(creneau_droit));     // utilise la méthode "writeMicroseconds" de la classe Servo pour générer les créneaux en sortie pour vitesse moteur droit
  
  //*********************** envoi sur le port série pour tests, debug et affichage  **********************************
  if (nb_cycles >= 10) {                        // l'affichage ne se produira que tous les 10 cycles de la boucle principale
Serial.print(commande_unitarisee);
Serial.print(" , ");
    //Serial.print(mesure_angle_pivot));
    //Serial.print(" , ");
    Serial.print(ecart);
    Serial.print(" , ");
    //  Serial.print(int(commande));
    //  Serial.print(" , ");
    //Serial.print(commande_unitarisee); 
    //  Serial.print(" , ");
    Serial.print(int(creneau_gauche));        // envoi de la valeur en microsecondes de la largeur d'impulson sur le port série (pour contrôle)
    Serial.print(" , ");
    Serial.print(int(creneau_droit));        // envoi de la valeur en microsecondes de la largeur d'impulson sur le port série (pour contrôle)
    Serial.println();
    nb_cycles = 0;                           // réinitialisation pour le comptage de cycles après affichage 
  } // fin if
  nb_cycles = nb_cycles + 1;                 // incrémentation pour le comptage de cycles avant affichage

  /* ************* attente du temps de cycle fixé  ********************** */
  // c'est ici que se gère la période d'échantillonnage choisie
  do {
    temps_cycle_calcule = micros() - temps_debut_cycle;
  } while (temps_cycle_calcule < (periode_echantillonnage * 1000));
  // fin do-while
  // Serial.println(temps_cycle_calcule);
} // fin de la boucle loop



